
#include <usb.h>
#include <string.h>
#include "jtag.h"

/*
  commands:

  0 -  39 : shift 1 - 40 bits w/TMS                                           <data>
 40 -  79 : shift 1 - 40 bits wo/TMS                                          <data>
 80 - 119 : shift 1 - 40 bits w/TMS and read                                  <data>
120 - 159 : shift 1 - 40 bits wo/TMS and read                                 <data>
160 - 221 : shift 1 - 5 inline bits w/TMS
222       : shift TDI and TMS 1-255 pairs                                     <count8>, <data>
223       : shift TDI and TMS 1-255 pairs and read                            <count8>, <data>
224       : shift 1-255 zeros w/TMS                                           <count8>
225       : shift 1-255 ones w/TMS                                            <count8>
226       : shift 1-255 zeros wo/TMS                                          <count8>
227       : shift 1-255 ones wo/TMS                                           <count8>
228       : shift 1-255 zeros w/TMS and read                                  <count8>
229       : shift 1-255 ones w/TMS and read                                   <count8>
230       : shift 1-255 zeros wo/TMS and read                                 <count8>
231       : shift 1-255 ones wo/TMS and read                                  <count8>
232 - 239 : send TDI/TMS and read TDO (1 bit) if bit 2 set
240 - 255 : shift 4 bits w/TMS

*/

enum {
  USB_VENDOR_REQUEST__CONFIGURE_XLATORX__SET_MASKS,
  USB_VENDOR_REQUEST__CONFIGURE_XLATORX__SET_STATUS,
  USB_VENDOR_REQUEST__CONFIGURE_XLATORX__JTAG_INIT1,
  USB_VENDOR_REQUEST__CONFIGURE_XLATORX__JTAG_INIT2,
  USB_VENDOR_REQUEST__CONFIGURE_XLATORX__JTAG_DEINIT
};


int jtag_init(jtag_commands* buf, struct usb_dev_handle* handle, unsigned char tdi_io, unsigned char tms_io, unsigned char tck_io, unsigned char tdo_io, unsigned char delay) {
  int ret;
  buf->commands_len = 0;
  buf->resp_bits = 0;
  buf->handle = handle;
  // set JTAG clock delay
  ret = usb_control_msg(handle, USB_TYPE_VENDOR | USB_RECIP_DEVICE, USB_VENDOR_REQUEST__CONFIGURE_XLATORX__SET_STATUS, delay, 0, (char*)0, 0, 100);
//  printf("init1 returned %d\n", ret);
  if( ret < 0 )
    return ret;
  ret = usb_control_msg(handle, USB_TYPE_VENDOR | USB_RECIP_DEVICE, USB_VENDOR_REQUEST__CONFIGURE_XLATORX__JTAG_INIT1, (1<<(tdi_io-1))|(256<<(tms_io-1)), 0, (char*)0, 0, 100);
//  printf("init2 returned %d\n", ret);
  if( ret < 0 )
    return ret;
  return usb_control_msg(handle, USB_TYPE_VENDOR | USB_RECIP_DEVICE, USB_VENDOR_REQUEST__CONFIGURE_XLATORX__JTAG_INIT2, (1<<(tck_io-1))|(256<<(tdo_io-1)), 0, (char*)0, 0, 100);
}

void jtag_reinit(jtag_commands* buf) {
  buf->commands_len = 0;
  buf->resp_bits = 0;
}

int jtag_deinit(jtag_commands* buf) {
  return usb_control_msg(buf->handle, USB_TYPE_VENDOR | USB_RECIP_DEVICE, USB_VENDOR_REQUEST__CONFIGURE_XLATORX__JTAG_DEINIT, 0, 0, (char*)0, 0, 10);
}

int jtag_run_commands(jtag_commands* buf) {
  int ret;
  if( buf->commands_len ) {
    ret = usb_bulk_write(buf->handle, 3, (char*)buf->commands, buf->commands_len, 250*buf->commands_len);
//  printf("write returned %d\n", ret);
    buf->commands_len = 0;
    if( ret < 0 )
      return ret;
    if( buf->resp_bits ) {
      ret = usb_bulk_read(buf->handle, 4, (char*)buf->responses, (buf->resp_bits+7)>>3, 1000);
//  printf("read returned %d\n", ret);
      if( ret < 0 )
        return ret;
      buf->resp_offset = 0;
      return buf->resp_bits;
    }
    return 0;
  } else {
    return -1;
  }
}

int jtag_get_response(jtag_commands* buf, unsigned char* to_here, unsigned long num_bits) {
  if( num_bits == 0 )
    return 0;
  if( num_bits > buf->resp_bits - buf->resp_offset )
    num_bits = buf->resp_bits - buf->resp_offset;
  if( !(buf->resp_offset&7) ) {
    memcpy(to_here, buf->responses+(buf->resp_offset>>3), (num_bits+7)>>3);
    buf->resp_offset += num_bits;
  } else {
    unsigned long i;
    for( i = 0; i < num_bits; i += 8 ) {
      to_here[0] = (buf->responses[buf->resp_offset>>3]>>(buf->resp_offset&7));
      if( num_bits-i > 8-buf->resp_offset )
        to_here[0] |= (buf->responses[(buf->resp_offset>>3)+1]<<(8-(buf->resp_offset&7)));
      if( num_bits-i >= 8 )
        buf->resp_offset += 8;
      else
        buf->resp_offset += num_bits-i;
    }
  }
  if( buf->resp_offset == buf->resp_bits )
    buf->resp_offset = buf->resp_bits = 0;
  return num_bits;
}

inline int jtag_push_byte(jtag_commands* buf, unsigned char byte) {
  if( buf->commands_len < sizeof(buf->commands) ) {
    buf->commands[buf->commands_len++] = byte;
    return 0;
  } else {
    return -1;
  }
}

int jtag_shift_bits_dir(jtag_commands* buf, unsigned char data, unsigned long num_bits, unsigned char final, unsigned char read) {
  if( num_bits > 8 )
    return -1;
  return jtag_shift_bits(buf, &data, num_bits, final, read);
}
int jtag_shift_bits(jtag_commands* buf, unsigned char* data, unsigned long num_bits, unsigned char final, unsigned char read) {
  int ret;
  if( num_bits < 1 || num_bits > 4096 )
    return -1;

  if( read )
    buf->resp_bits += num_bits;
  while( num_bits ) {
    if( num_bits <= 40 ) {
      if( num_bits == 4 && final && !read ) {
        return jtag_push_byte(buf, 240+(data[0]&15));
      }
      ret = jtag_push_byte(buf, (final ? read ? 80 : 0 : read ? 120 : 40) + (num_bits-1));
      if( ret < 0 )
        return ret;
      while( num_bits ) {
        ret = jtag_push_byte(buf, data[0]);
        if( ret < 0 )
          return ret;
        ++data;
        if( num_bits > 8 )
          num_bits -= 8;
        else
          num_bits = 0;
      }
    } else {
      int i;
      ret = jtag_push_byte(buf, (read ? 120 : 40) + 39);
      if( ret < 0 )
        return ret;
      for( i = 0; i < 5; ++i ) {
        ret = jtag_push_byte(buf, data[0]);
        if( ret < 0 )
          return ret;
        ++data;
        num_bits -= 8;
      }
    }
  }
  return 0;
}

int jtag_shift_TMS(jtag_commands* buf, unsigned char data, unsigned char num_bits) {
  int ret;
  if( num_bits < 1 || num_bits > 32 )
    return -1;

  while( num_bits ) {
    if( num_bits <= 5 ) {
      switch(num_bits) {
      case 1:
        return jtag_push_byte(buf, 160+(data&1));
      case 2:
        return jtag_push_byte(buf, 162+(data&3));
      case 3:
        return jtag_push_byte(buf, 166+(data&7));
      case 4:
        return jtag_push_byte(buf, 174+(data&15));
      default: // case 5
        return jtag_push_byte(buf, 190+(data&31));
      }
    } else {
      ret = jtag_push_byte(buf, 190+(data&31));
      if( ret < 0 )
        return ret;
      data >>= 5;
      num_bits -= 5;
    }
  }
  return 0;
}

// TDI, TMS, TDI, TMS, ...
int jtag_send_data_dir(jtag_commands* buf, unsigned char two_bit_data, unsigned long num_bit_pairs, unsigned char read) {
  if( num_bit_pairs > 4 )
    return -1;
  return jtag_send_data(buf, &two_bit_data, num_bit_pairs, read);
}
int jtag_send_data(jtag_commands* buf, unsigned char* two_bit_data, unsigned long num_bit_pairs, unsigned char read) {
  int ret;
  if( num_bit_pairs < 1 || num_bit_pairs > 4096 )
    return -1;

  if( read )
    buf->resp_bits += num_bit_pairs;

  while( num_bit_pairs ) {
    if( num_bit_pairs <= 255 ) {
      if( num_bit_pairs == 1 ) {
        return jtag_push_byte(buf, 232+(two_bit_data[0]&3)+(read ? 4 : 0));
      } else {
        ret = jtag_push_byte(buf, read ? 223 : 222);
        if( ret < 0 )
          return ret;
        ret = jtag_push_byte(buf, num_bit_pairs);
        if( ret < 0 )
          return ret;
        while( num_bit_pairs ) {
          ret = jtag_push_byte(buf, two_bit_data[0]);
          if( ret < 0 )
            return ret;
          ++two_bit_data;
          if( num_bit_pairs > 4 )
            num_bit_pairs -= 4;
          else
            num_bit_pairs = 0;
        }
      }
    } else {
      int i;
      ret = jtag_push_byte(buf, read ? 223 : 222);
      if( ret < 0 )
        return ret;
      ret = jtag_push_byte(buf, 252);
      if( ret < 0 )
        return ret;
      for( i = 0; i < 252/4; ++i ) {
        ret = jtag_push_byte(buf, two_bit_data[0]);
        if( ret < 0 )
          return ret;
        ++two_bit_data;
        num_bit_pairs -= 4;
      }
    }
  }
  return 0;
}

int jtag_shift_zeros(jtag_commands* buf, unsigned long num_bits, unsigned char final, unsigned char read) {
  int ret;
  if( num_bits < 1 || num_bits > 4096 )
    return -1;

  if( read )
    buf->resp_bits += num_bits;

  while( num_bits ) {
    if( num_bits <= 255 ) {
      ret = jtag_push_byte(buf, 224+(final ? 0 : 2)+(read ? 4 : 0)+0);
      if( ret < 0 )
        return ret;
      return jtag_push_byte(buf, num_bits);
    } else {
      int i;
      ret = jtag_push_byte(buf, 224+(read ? 4 : 0)+0);
      if( ret < 0 )
        return ret;
      ret = jtag_push_byte(buf, 255);
      if( ret < 0 )
        return ret;
      num_bits -= 255;
    }
  }
  return 0;
}

int jtag_shift_ones(jtag_commands* buf, unsigned long num_bits, unsigned char final, unsigned char read) {
  int ret;
  if( num_bits < 1 || num_bits > 4096 )
    return -1;

  if( read )
    buf->resp_bits += num_bits;

  while( num_bits ) {
    if( num_bits <= 255 ) {
      ret = jtag_push_byte(buf, 224+(final ? 0 : 2)+(read ? 4 : 0)+1);
      if( ret < 0 )
        return ret;
      return jtag_push_byte(buf, num_bits);
    } else {
      int i;
      ret = jtag_push_byte(buf, 224+(read ? 4 : 0)+1);
      if( ret < 0 )
        return ret;
      ret = jtag_push_byte(buf, 255);
      if( ret < 0 )
        return ret;
      num_bits -= 255;
    }
  }
  return 0;
}


#define BITS1(a)               (a)
#define BITS2(a,b)             ((a)|((b)<<1))
#define BITS3(a,b,c)           ((a)|((b)<<1)|((c)<<2))
#define BITS4(a,b,c,d)         ((a)|((b)<<1)|((c)<<2)|((d)<<3))
#define BITS5(a,b,c,d,e)       ((a)|((b)<<1)|((c)<<2)|((d)<<3)|((e)<<4))
#define BITS6(a,b,c,d,e,f)     ((a)|((b)<<1)|((c)<<2)|((d)<<3)|((e)<<4)|((f)<<5))
#define BITS7(a,b,c,d,e,f,g)   ((a)|((b)<<1)|((c)<<2)|((d)<<3)|((e)<<4)|((f)<<5)|((g)<<6))
#define BITS8(a,b,c,d,e,f,g,h) ((a)|((b)<<1)|((c)<<2)|((d)<<3)|((e)<<4)|((f)<<5)|((g)<<6)|((h)<<7))
#define TMS1(a)               BITS1(a), 1
#define TMS2(a,b)             BITS2(a,b), 2
#define TMS3(a,b,c)           BITS3(a,b,c), 3
#define TMS4(a,b,c,d)         BITS4(a,b,c,d), 4
#define TMS5(a,b,c,d,e)       BITS5(a,b,c,d,e), 5
#define TMS6(a,b,c,d,e,f)     BITS6(a,b,c,d,e,f), 6
#define TMS7(a,b,c,d,e,f,g)   BITS7(a,b,c,d,e,f,g), 7
#define TMS8(a,b,c,d,e,f,g,h) BITS8(a,b,c,d,e,f,g,h), 8

int jtag_reset_TAP(jtag_commands* buf) {
  return jtag_shift_TMS(buf, TMS5(1, 1, 1, 1, 1));
}

int jtag_RESET_to_IDLE(jtag_commands* buf) {
  return jtag_shift_TMS(buf, TMS1(0));
}

int jtag_IDLE_to_SHIFT_DR(jtag_commands* buf) {
  return jtag_shift_TMS(buf, TMS3(1, 0, 0));
}

int jtag_IDLE_to_SHIFT_IR(jtag_commands* buf) {
  return jtag_shift_TMS(buf, TMS4(1, 1, 0, 0));
}

int jtag_EXIT_DR_to_SHIFT_DR_via_UPDATE_DR(jtag_commands* buf) {
  return jtag_shift_TMS(buf, TMS6(0, 1, 1, 1, 0, 0));
}

int jtag_EXIT_DR_to_SHIFT_IR_via_UPDATE_DR(jtag_commands* buf) {
  return jtag_shift_TMS(buf, TMS7(0, 1, 1, 1, 1, 0, 0));
}

int jtag_EXIT_DR_to_SHIFT_DR_direct(jtag_commands* buf) {
  return jtag_shift_TMS(buf, TMS3(0, 1, 0));
}

int jtag_EXIT_IR_to_SHIFT_IR_via_UPDATE_IR(jtag_commands* buf) {
  return jtag_shift_TMS(buf, TMS7(0, 1, 1, 1, 1, 0, 0));
}

int jtag_EXIT_IR_to_SHIFT_DR_via_UPDATE_IR(jtag_commands* buf) {
  return jtag_shift_TMS(buf, TMS6(0, 1, 1, 1, 0, 0));
}

int jtag_EXIT_IR_to_SHIFT_IR_direct(jtag_commands* buf) {
  return jtag_shift_TMS(buf, TMS3(0, 1, 0));
}
